import java.awt.*;
import javax.swing.*;
import java.awt.event.*;
import javax.media.*;
import javax.media.format.*;
/* Version 1.1, Illah added API description in code, got rid of
    newline / return problem in .txt files, and fixed .txt typos */
/* Version 1.0, JAVA library for using USB camera in Java
    Released October 2003 */
/* To use this file you must have JMF installed on your computer.
    Note that JMF should be installed after you have installed
    whatever camera you're using (it detects your drivers on-board
    during installation).  Alternatively you can run the "jmfinit" program
    that is distributed with the JMF to re-detect the available cameras.
    The API described below allows you to initialize / get the camera,
    snap pictures with it, display the pictures on-screen,
    draw rectangles in the picture to identify things you've found
    through image analysis, and of course obtain
    RGB values for any pixel in the image. The API follows:

    UsbCamera.getInstance() returns a object of type UsbCamera
    do this first to ensure you have a camera; check to make sure
    the return object is not NULL.

    cam.setVisible(true) sets the UsbCamera to show the image on-screen
    If you do not call this function, you still get images, but you
    don't display them on-screen.

    cam.snap() is the function that actually fills memory with a
    new image. Don't do this more than 10 times per second or you
    build up a weird lag between real-time and image-time.

    Note that snap() actually calls getFrame() and then draw();
    To speed things up, just call getFrame() and avoid draw() and
    of course then don't use snap() at all either.

    Accessing pixel values can be done in two ways. The slow but
    easy way is to use getPixel(x,y), which returns an int[].
    In this array, you can access the individual colors using
    RED, GREEN and BLUE static values as follows:
        <example>

        int [] pixel = cam.getPixel(50,50);
        System.out.println("Red is: " + pixel[UsbCamera.RED]);

   A much faster method uses getRawPixel() and macro's like
   getRed():

        int val = cam.getRawPixel(50,50);
        int red = UsbCamera.getRed(val);

    To demonstrate the ability to capture and show an image and then
    make changes directly on the image, we have a helper function
    for drawing rectangles on-image: setRect(x,y,width, height) will draw
    a rectangle.  Note that x,y is the upper left corner of the
    rectangle.  Also note unsetRect() for removing the rectangle
    you have drawn. You can only draw one rectangle on screen with this
    demo function.

    Useful variables/constants:
        cam.XSIZE and cam.YSIZE (see the sample program)
        The upper left pixel is (0,0) and x is positive to
        positive pixels are down and to the right.
    */



public class UsbCamera extends JFrame {
    public static final int ALPHA = 0;
    public static final int RED = 1;
    public static final int GREEN = 2;
    public static final int BLUE = 3;

    public static final int XSIZE = 320;
    public static final int YSIZE = 260;

    private static UsbCamera theInstance;

    private int[] pixels;
    private ImageProcessor imp;
    private Player player;
    private Image lastPic;
    private PicCanvas canvas;

    /* Ensure class is singleton */
    public synchronized static UsbCamera getInstance() {
	if(theInstance == null) {
	    theInstance = new UsbCamera();
	}
	return theInstance;
    }

    private UsbCamera() {
	super();

	imp = new ImageProcessor(XSIZE,YSIZE);
	VideoFormat vFormat = new YUVFormat();
	player = imp.initPlayer(vFormat);

	canvas = new PicCanvas();
	getContentPane().add(canvas, BorderLayout.CENTER);

	this.setSize(XSIZE,YSIZE);

	/* Wait for init of stream */
	for(int i = 0; i < 20; i++) {
	    try { Thread.sleep(100); } catch (InterruptedException e) {}
	    getFrame();
	    if(lastPic != null) break;
	}

	if(lastPic != null) {
	    draw();
	} else {
	    System.err.println("Could not init camera.");
	}
    }

    /* External API */
    /* Take a picture */
    public void snap() {
	getFrame();
	draw();
    }

    public void getFrame() {
	lastPic = imp.getFrame(player);
	pixels = null;
    }

    /* Display a picture given a PicCanvas object */
    public void draw() {
	if(lastPic == null) {
	    System.err.println("Take a picture before drawing it!");
	    return;
	}

	canvas.setParams(lastPic, 0, 0);
    }

    /* Get a pixel value */
    public int[] getPixel(int x, int y) {
	int pixel;
	try {
	    pixel = getRawPixel(x,y);
	} catch (Exception e) {
	    return null;
	}

	int retval[] = new int[4];

	retval[ALPHA] = getAlpha(pixel);
	retval[RED] = getRed(pixel);
	retval[GREEN] = getGreen(pixel);
	retval[BLUE] = getBlue(pixel);

	return retval;
    }

    public int getRawPixel(int x, int y) throws Exception {
	if(lastPic == null
	   || x > XSIZE || y > YSIZE
	   || x < 0 || y < 0) {
	    throw new Exception("bad pixel");
	}

	if(lastPic != null && pixels == null) {
	    do {
		try {
		    pixels = imp.getPixelBlock(lastPic);
		} catch(InterruptedException e) {}
	    } while(pixels == null);
	}

	return pixels[y * XSIZE + x];
    }

    public static int getAlpha(int pixel) {
	return (pixel >> 24) & 0xff;
    }

    public static int getRed(int pixel) {
	return (pixel >> 16) & 0xff;
    }

    public static int getGreen(int pixel) {
	return (pixel >> 8) & 0xff;
    }

    public static int getBlue(int pixel) {
	return pixel & 0xff;
    }

    /* Return the full Image object from the last picture */
    public Image getImage() {
	return lastPic;
    }

    public void setRect(double x, double y, double xp, double yp) {
	canvas.setRect(x, y, xp, yp);
    }

    public void unsetRect() {
	canvas.unsetRect();
    }
}
